/**
 * Minimal Klondike Solitaire Logic & UI
 * Handles game state, rules, and rendering within a Shadow DOM.
 */

const CSS_STYLES = `
:host {
  all: initial;
  --bg-color: #202124;
  --text-color: #e8eaed;
  --card-bg: #505458;
  --card-border: #5f6368;
  --slot-bg: rgba(255,255,255,0.05);
  --slot-border: #5f6368;
  --highlight: #8ab4f8;
  --btn-hover: rgba(255,255,255,0.1);
  --scrollbar: #9aa0a6;
  
  --color-red: #ff8a80;
  --color-orange: #ffd180;
  --color-white: #ffffff;
  --color-gray: #bdc1c6;

  font-family: Arial, sans-serif;
  font-size: 12px;
  line-height: 1;
  display: block;
  position: fixed;
  top: 0;
  left: 0;
  width: 100%;
  z-index: 2147483647;
  pointer-events: none; /* Let clicks pass through when not hitting UI */
}

:host(.light-theme) {
  --bg-color: #f1f3f4;
  --text-color: #202124;
  --card-bg: #ffffff;
  --card-border: #dadce0;
  --slot-bg: rgba(0,0,0,0.05);
  --slot-border: #dadce0;
  --highlight: #1a73e8;
  --btn-hover: rgba(0,0,0,0.1);
  --scrollbar: #5f6368;
  
  --color-red: #d93025;
  --color-orange: #e37400;
  --color-white: #000000;
  --color-gray: #5f6368;
}

.overlay-container {
  width: 100%;
  background-color: var(--bg-color);
  color: var(--text-color);
  box-shadow: 0 2px 4px rgba(0,0,0,0.5);
  display: flex;
  align-items: center;
  box-sizing: border-box;
  pointer-events: auto; /* Capture clicks on the bar */
  user-select: none;
  overflow: hidden;
  transition: height 0.2s;
}

.overlay-container.minimized {
  height: 2px;
  cursor: pointer;
  background-color: var(--card-border);
}

.overlay-container.normal {
  height: 30px;
  padding: 0 8px;
}

.overlay-container.minimized > * {
  display: none;
}

/* Layout Sections */
.section {
  display: flex;
  align-items: center;
  gap: 4px;
  height: 100%;
}

.spacer {
  flex: 1;
}

.divider {
  width: 1px;
  height: 16px;
  background-color: var(--card-border);
  margin: 0 8px;
}

/* Card / Slot Styling */
.slot {
  width: 24px;
  height: 24px;
  border: 1px dashed var(--slot-border);
  border-radius: 3px;
  display: flex;
  align-items: center;
  justify-content: center;
  cursor: pointer;
  position: relative;
  background: var(--slot-bg);
}

.slot:hover {
  background: var(--btn-hover);
}

.slot.selected {
  border-color: var(--highlight);
  box-shadow: 0 0 0 1px var(--highlight);
}

.card {
  width: 22px;
  height: 22px;
  background-color: var(--card-bg);
  border-radius: 2px;
  display: flex;
  align-items: center;
  justify-content: center;
  font-weight: bold;
  font-size: 11px;
  border: 1px solid var(--card-border);
  font-family: Arial, sans-serif;
}

/* Colors */
.color-red { color: var(--color-red); }
.color-orange { color: var(--color-orange); }
.color-white { color: var(--color-white); }
.color-gray { color: var(--color-gray); }

/* Indicators */
.badge {
  font-size: 9px;
  color: var(--scrollbar);
  margin-right: 2px;
  min-width: 10px;
  text-align: center;
}

.stock-count, .waste-count {
  font-size: 10px;
  color: var(--scrollbar);
  margin-right: 4px;
  width: 18px;
  text-align: right;
}

/* Buttons */
.btn {
  background: none;
  border: 1px solid var(--card-border);
  color: var(--text-color);
  border-radius: 3px;
  cursor: pointer;
  font-size: 10px;
  padding: 2px 6px;
  margin-left: 4px;
}
.btn:hover { background: var(--btn-hover); }

/* Tableau Overlap */
.tableau-pile {
  display: flex;
  align-items: center;
}
.tableau-card {
  margin-left: -14px; /* Overlap cards */
  position: relative;
  transition: margin-left 0.1s;
  justify-content: flex-start !important; /* Keep rank visible on left */
  padding-left: 3px !important;
}
`;

// --- Constants & Utils ---

const SUITS = {
  RED: 'red',
  ORANGE: 'orange',
  WHITE: 'white',
  GRAY: 'gray'
};

const COLORS = [SUITS.RED, SUITS.ORANGE, SUITS.WHITE, SUITS.GRAY];

// Group 0: Red/Orange. Group 1: White/Gray.
const getColorGroup = (color) => (color === SUITS.RED || color === SUITS.ORANGE) ? 0 : 1;

const RANKS = ['1', '2', '3', '4', '5', '6', '7', '8', '9', '10', '11', '12', '13'];

const generateDeck = () => {
  let deck = [];
  let id = 0;
  COLORS.forEach(color => {
    for (let r = 1; r <= 13; r++) {
      deck.push({
        id: id++,
        rank: r,
        color: color,
        faceUp: false
      });
    }
  });
  return deck;
};

const shuffle = (array) => {
  const newArr = [...array];
  for (let i = newArr.length - 1; i > 0; i--) {
    const j = crypto.getRandomValues(new Uint32Array(1))[0] % (i + 1);
    [newArr[i], newArr[j]] = [newArr[j], newArr[i]];
  }
  return newArr;
};

// --- Game Class ---

class SolitaireGame {
  constructor(rootElement) {
    this.root = rootElement;
    this.state = this.getInitialState();
    this.history = []; // Undo stack
    this.minimized = false;
    this.lightTheme = false;
    
    // UI Elements
    this.container = document.createElement('div');
    this.container.className = 'overlay-container normal';
    this.root.appendChild(this.container);
    
    this.render();
  }

  getInitialState() {
    const deck = shuffle(generateDeck());
    const tableau = [[], [], [], [], [], [], []];
    
    // Deal
    for (let i = 0; i < 7; i++) {
      for (let j = 0; j <= i; j++) {
        const card = deck.pop();
        if (j === i) card.faceUp = true;
        tableau[i].push(card);
      }
    }

    return {
      stock: deck, // Top is end of array
      waste: [],   // Top is end of array
      tableau: tableau,
      foundation: {
        [SUITS.RED]: [],
        [SUITS.ORANGE]: [],
        [SUITS.WHITE]: [],
        [SUITS.GRAY]: []
      },
      selection: null // { type: 'waste'|'tableau', index: int, cardIndex: int (for tableau runs) }
    };
  }

  saveState() {
    // Deep copy state for undo
    this.history.push(JSON.stringify(this.state));
  }

  restoreState() {
    if (this.history.length === 0) return;
    const prevState = JSON.parse(this.history.pop());
    this.state = prevState;
    this.render();
  }

  restart() {
    this.history = [];
    this.state = this.getInitialState();
    this.render();
  }

  // --- Logic ---

  // Helper: Get top card of a pile
  getTop(pile) {
    return pile.length > 0 ? pile[pile.length - 1] : null;
  }

  // Helper: Check if move is valid to Foundation
  canMoveToFoundation(card, color) {
    if (card.color !== color) return false;
    const pile = this.state.foundation[color];
    const top = this.getTop(pile);
    const targetRank = top ? top.rank : 0;
    return card.rank === targetRank + 1;
  }

  // Helper: Check if move is valid to Tableau
  canMoveToTableau(card, tableauIdx) {
    const pile = this.state.tableau[tableauIdx];
    const top = this.getTop(pile);
    
    if (!top) {
      return card.rank === 13; // King only on empty
    }
    
    return top.faceUp && 
           (card.rank === top.rank - 1) && 
           (getColorGroup(card.color) !== getColorGroup(top.color));
  }

  // Action: Stock Click
  handleStockClick() {
    this.saveState();
    if (this.state.stock.length === 0) {
      // Recycle waste
      this.state.stock = this.state.waste.reverse().map(c => ({ ...c, faceUp: false }));
      this.state.waste = [];
    } else {
      // Draw 3
      const count = Math.min(3, this.state.stock.length);
      for (let i = 0; i < count; i++) {
        const card = this.state.stock.pop();
        card.faceUp = true;
        this.state.waste.push(card);
      }
    }
    this.state.selection = null;
    this.render();
  }

  // Action: Auto Place (from Waste or Tableau Top)
  autoPlace(card, sourceCallback) {
    // 1. Try Foundation
    for (const color of COLORS) {
      if (this.canMoveToFoundation(card, color)) {
        this.saveState();
        sourceCallback(); // Remove from source
        this.state.foundation[color].push(card);
        this.checkFlipAfterMove();
        this.state.selection = null;
        this.render();
        return true;
      }
    }
    
    // 2. Try Tableau
    for (let i = 0; i < 7; i++) {
      if (this.canMoveToTableau(card, i)) {
        this.saveState();
        sourceCallback();
        this.state.tableau[i].push(card);
        this.checkFlipAfterMove();
        this.state.selection = null;
        this.render();
        return true;
      }
    }
    return false;
  }

  // Action: Auto Move Run (Tableau to Tableau)
  autoMoveRun(sourceTableauIdx, cardIdx) {
    const pile = this.state.tableau[sourceTableauIdx];
    const run = pile.slice(cardIdx);
    const headCard = run[0];

    for (let i = 0; i < 7; i++) {
      if (i === sourceTableauIdx) continue;
      if (this.canMoveToTableau(headCard, i)) {
        this.saveState();
        pile.splice(cardIdx, run.length);
        this.state.tableau[i].push(...run);
        this.checkFlipAfterMove();
        this.state.selection = null;
        this.render();
        return true;
      }
    }
    return false;
  }

  // Action: Move Selection to Target
  attemptMoveToTableau(targetIdx) {
    if (!this.state.selection) return;
    
    const sel = this.state.selection;
    let cardsToMove = [];
    
    // Gather cards
    if (sel.type === 'waste') {
      cardsToMove = [this.getTop(this.state.waste)];
    } else if (sel.type === 'tableau') {
      const pile = this.state.tableau[sel.index];
      cardsToMove = pile.slice(sel.cardIndex);
    }

    if (cardsToMove.length === 0 || !cardsToMove[0]) return;

    const leadCard = cardsToMove[0];

    if (this.canMoveToTableau(leadCard, targetIdx)) {
      this.saveState();
      
      // Remove from source
      if (sel.type === 'waste') {
        this.state.waste.pop();
      } else {
        const srcPile = this.state.tableau[sel.index];
        srcPile.splice(sel.cardIndex, cardsToMove.length);
      }
      
      // Add to target
      this.state.tableau[targetIdx].push(...cardsToMove);
      
      this.checkFlipAfterMove();
      this.state.selection = null;
      this.render();
    } else {
      // Invalid move, just deselect
      this.state.selection = null;
      this.render();
    }
  }

  // Action: Move Selection to Foundation (Single card only)
  attemptMoveToFoundation(color) {
    if (!this.state.selection) return;
    
    const sel = this.state.selection;
    let card = null;

    // Can only move single card to foundation
    if (sel.type === 'waste') {
      card = this.getTop(this.state.waste);
    } else if (sel.type === 'tableau') {
      const pile = this.state.tableau[sel.index];
      // Must be the top card
      if (sel.cardIndex === pile.length - 1) {
        card = pile[sel.cardIndex];
      }
    }

    if (card && this.canMoveToFoundation(card, color)) {
      this.saveState();
      
      if (sel.type === 'waste') {
        this.state.waste.pop();
      } else {
        this.state.tableau[sel.index].pop();
      }
      
      this.state.foundation[color].push(card);
      this.checkFlipAfterMove();
      this.state.selection = null;
      this.render();
    }
  }

  checkFlipAfterMove() {
    // Check all tableaus. If top is face down, flip it.
    this.state.tableau.forEach(pile => {
      const top = this.getTop(pile);
      if (top && !top.faceUp) {
        top.faceUp = true;
      }
    });
  }

  // --- Rendering ---

  renderCard(card, onClick, isSelected = false) {
    const el = document.createElement('div');
    el.className = `card color-${card.color}`;
    el.textContent = `${RANKS[card.rank - 1]}`;
    if (isSelected) {
      el.style.borderColor = 'var(--highlight)';
      el.style.boxShadow = '0 0 0 1px var(--highlight)';
    }
    if (onClick) el.onclick = (e) => { e.stopPropagation(); onClick(); };
    return el;
  }

  renderSlot(content, onClick, isSelected = false, badgeText = null) {
    const slot = document.createElement('div');
    slot.className = `slot ${isSelected ? 'selected' : ''}`;
    slot.onclick = onClick;

    if (badgeText) {
      const badge = document.createElement('span');
      badge.className = 'badge';
      badge.textContent = badgeText;
      // Insert badge before slot content in the parent usually, but here we are inside slot?
      // Layout: [Badge][Slot] is better.
      // Let's return a wrapper if badge exists.
      const wrapper = document.createElement('div');
      wrapper.style.display = 'flex';
      wrapper.style.alignItems = 'center';
      wrapper.appendChild(badge);
      if (content) slot.appendChild(content);
      wrapper.appendChild(slot);
      return wrapper;
    }

    if (content) slot.appendChild(content);
    return slot;
  }

  render() {
    this.container.innerHTML = '';
    
    if (this.minimized) {
      this.container.className = 'overlay-container minimized';
      this.container.onclick = () => {
        this.minimized = false;
        this.render();
      };
      return;
    }
    this.container.className = 'overlay-container normal';
    this.container.onclick = null;

    if (this.lightTheme) {
      this.root.host.classList.add('light-theme');
    } else {
      this.root.host.classList.remove('light-theme');
    }

    // 1. Stock
    const stockCount = document.createElement('span');
    stockCount.className = 'stock-count';
    stockCount.textContent = this.state.stock.length;
    this.container.appendChild(stockCount);

    const stockSlot = document.createElement('div');
    stockSlot.className = 'slot';
    stockSlot.style.background = this.state.stock.length > 0 ? 'var(--card-border)' : 'transparent';
    stockSlot.onclick = () => this.handleStockClick();
    this.container.appendChild(stockSlot);

    this.container.appendChild(this.createDivider());

    // 2. Waste
    const wasteTop = this.getTop(this.state.waste);
    const isWasteSelected = this.state.selection && this.state.selection.type === 'waste';
    
    const wasteContent = wasteTop ? this.renderCard(wasteTop, () => {
      // Click Waste Card
      if (this.state.selection && this.state.selection.type === 'waste') {
        // Deselect
        this.state.selection = null;
        this.render();
      } else {
        // Select or Auto-place
        // If we want auto-place on single click:
        const placed = this.autoPlace(wasteTop, () => this.state.waste.pop());
        if (!placed) {
          // Select it for manual move
          this.state.selection = { type: 'waste', index: 0 };
          this.render();
        }
      }
    }, isWasteSelected) : null;

    const wasteCount = document.createElement('span');
    wasteCount.className = 'waste-count';
    wasteCount.textContent = this.state.waste.length;
    this.container.appendChild(wasteCount);

    this.container.appendChild(this.renderSlot(wasteContent, () => {}, isWasteSelected, null));

    this.container.appendChild(this.createDivider());

    // 3. Tableau Area
    this.state.tableau.forEach((pile, idx) => {
        // Normal View (Overlapping)
        const pileContainer = document.createElement('div');
        pileContainer.className = 'tableau-pile';

        const faceDownCount = pile.reduce((acc, c) => acc + (c.faceUp ? 0 : 1), 0);
        
        // Base Slot (Empty or Face-down indicator)
        const baseSlot = this.renderSlot(null, () => {
           if (this.state.selection && (this.state.selection.index !== idx)) {
             this.attemptMoveToTableau(idx);
           }
        }, false, faceDownCount > 0 ? `+${faceDownCount}` : '');
        baseSlot.classList.add('tableau-base');

        pileContainer.appendChild(baseSlot);

        // Render Face-up Cards
        pile.forEach((card, cardIdx) => {
          if (!card.faceUp) return;

          const isSelected = this.state.selection && 
                             this.state.selection.type === 'tableau' && 
                             this.state.selection.index === idx &&
                             cardIdx >= this.state.selection.cardIndex;

          const cardEl = this.renderCard(card, () => {
            const isTopCard = (cardIdx === pile.length - 1);
            
            if (this.state.selection) {
               if (this.state.selection.type === 'tableau' && this.state.selection.index === idx) {
                 // Clicked same pile
                 this.state.selection = null;
                 this.render();
               } else {
                 // Move selection TO here
                 this.attemptMoveToTableau(idx);
               }
            } else {
              // No selection
              if (isTopCard) {
                const placed = this.autoPlace(card, () => pile.pop());
                if (!placed) {
                   this.state.selection = { type: 'tableau', index: idx, cardIndex: cardIdx };
                   this.render();
                }
              } else {
                // Select run
                const placed = this.autoMoveRun(idx, cardIdx);
                if (!placed) {
                  this.state.selection = { type: 'tableau', index: idx, cardIndex: cardIdx };
                  this.render();
                }
              }
            }
          }, isSelected);

          cardEl.classList.add('tableau-card');
          cardEl.style.zIndex = cardIdx + 1;
          pileContainer.appendChild(cardEl);
        });

        this.container.appendChild(pileContainer);
      // Small spacer
      const sp = document.createElement('div');
      sp.style.width = '2px';
      this.container.appendChild(sp);
    });

    this.container.appendChild(this.createDivider());

    // 4. Foundations
    COLORS.forEach(color => {
      const pile = this.state.foundation[color];
      const top = this.getTop(pile);
      const content = top ? this.renderCard(top, () => {}, false) : null;
      // Show rank letter if empty? No, just color slot.
      const slot = this.renderSlot(content, () => {
        this.attemptMoveToFoundation(color);
      });
      // Tint border for color hint
      slot.style.borderColor = this.getColorHex(color);
      this.container.appendChild(slot);
      const sp = document.createElement('div');
      sp.style.width = '2px';
      this.container.appendChild(sp);
    });

    this.container.appendChild(document.createElement('div')).className = 'spacer';

    // 5. Controls
    this.createBtn('Undo', () => this.restoreState());
    this.createBtn('Restart', () => this.restart());
    this.createBtn(this.lightTheme ? 'Dark' : 'Light', () => {
      this.lightTheme = !this.lightTheme;
      this.render();
    });
    this.createBtn('_', () => {
      this.minimized = true;
      this.render();
    });
  }

  createDivider() {
    const d = document.createElement('div');
    d.className = 'divider';
    return d;
  }

  createBtn(text, onClick) {
    const btn = document.createElement('button');
    btn.className = 'btn';
    btn.textContent = text;
    btn.onclick = onClick;
    this.container.appendChild(btn);
  }

  getColorHex(color) {
    switch(color) {
      case SUITS.RED: return 'var(--color-red)';
      case SUITS.ORANGE: return 'var(--color-orange)';
      case SUITS.WHITE: return 'var(--color-white)';
      case SUITS.GRAY: return 'var(--color-gray)';
      default: return 'var(--card-border)';
    }
  }
}

// Expose to global scope for content_script to use
window.SolitaireGame = SolitaireGame;
window.SolitaireCSS = CSS_STYLES;
